/*********************************************************************** 
 * timer.c
 * 
 * Project: LPC18xx/LPC43xx common drivers
 * 
 * Description: Timer Driver
 * 
 * Copyright(C) 2010, NXP Semiconductor
 * All rights reserved.
 * 
 ***********************************************************************
 * Software that is described herein is for illustrative purposes only  
 * which provides customers with programming information regarding the  
 * products. This software is supplied "AS IS" without any warranties.  
 * NXP Semiconductors assumes no responsibility or liability for the 
 * use of the software, conveys no license or title under any patent, 
 * copyright, or mask work right to the product. NXP Semiconductors 
 * reserves the right to make changes in the software without 
 * notification. NXP Semiconductors also make no representation or 
 * warranty that such application will be suitable for the specified 
 * use without further testing or modification. 
 **********************************************************************/
#ifdef LPC43XX
  #include "LPC43xx_TC2.h"
  #include "scu_LPC43xx.h"
#endif
#ifdef LPC18XX
  #include "LPC18xx.h"
  #include "scu_LPC18xx.h"
#endif

#include "type.h"
#include "timer.h"


volatile uint32_t timer0_m0_counter = 0;
volatile uint32_t timer1_m0_counter = 0;
volatile uint32_t timer2_m0_counter = 0;
volatile uint32_t timer3_m0_counter = 0;
volatile uint32_t timer0_m1_counter = 0;
volatile uint32_t timer1_m1_counter = 0;
volatile uint32_t timer2_m1_counter = 0;
volatile uint32_t timer3_m1_counter = 0;

volatile uint32_t timer0_capture0 = 0;
volatile uint32_t timer1_capture0 = 0;
volatile uint32_t timer2_capture0 = 0;
volatile uint32_t timer3_capture0 = 0;
volatile uint32_t timer0_capture1 = 0;
volatile uint32_t timer1_capture1 = 0;
volatile uint32_t timer2_capture1 = 0;
volatile uint32_t timer3_capture1 = 0;

/*****************************************************************************
** Function name:       delayMs
**
** Descriptions:        Start the timer delay in milli seconds
**                      until elapsed
**
** parameters:          timer number, Delay value in milli second            
**                      
** Returned value:      None
** 
*****************************************************************************/
void delayMs(uint8_t timer_num, uint32_t delayInMs)
{
  if ( timer_num == 0 )
  {
    LPC_TIMER0->TCR = 0x02;     /* reset timer */
    LPC_TIMER0->PR  = 0x00;     /* set prescaler to zero */
#ifdef LPC43XX  
    LPC_TIMER0->MR[0] = delayInMs * ((M4Frequency / 1000) - 1);
#endif
#ifdef LPC18XX  
    LPC_TIMER0->MR[0] = delayInMs * ((M3Frequency / 1000) - 1);
#endif
    LPC_TIMER0->IR  = 0xff;     /* reset all interrrupts */
    LPC_TIMER0->MCR = 0x04;     /* stop timer on match */
    LPC_TIMER0->TCR = 0x01;     /* start timer */
  
    /* wait until delay time has elapsed */
    while (LPC_TIMER0->TCR & 0x01);
  }
  else if ( timer_num == 1 )
  {
    LPC_TIMER1->TCR = 0x02;     /* reset timer */
    LPC_TIMER1->PR  = 0x00;     /* set prescaler to zero */
#ifdef LPC43XX
    LPC_TIMER1->MR[0] = delayInMs * ((M4Frequency / 1000) - 1);
#endif
#ifdef LPC18XX
    LPC_TIMER1->MR[0] = delayInMs * ((M3Frequency / 1000) - 1);
#endif
    LPC_TIMER1->IR  = 0xff;     /* reset all interrrupts */
    LPC_TIMER1->MCR = 0x04;     /* stop timer on match */
    LPC_TIMER1->TCR = 0x01;     /* start timer */
  
    /* wait until delay time has elapsed */
    while (LPC_TIMER1->TCR & 0x01);
  }
  else if ( timer_num == 2 )
  {
    LPC_TIMER2->TCR = 0x02;     /* reset timer */
    LPC_TIMER2->PR  = 0x00;     /* set prescaler to zero */
#ifdef LPC43XX
    LPC_TIMER2->MR[0] = delayInMs * ((M4Frequency / 1000) - 1);
#endif
#ifdef LPC18XX
    LPC_TIMER2->MR[0] = delayInMs * ((M3Frequency / 1000) - 1);
#endif
    LPC_TIMER2->IR  = 0xff;     /* reset all interrrupts */
    LPC_TIMER2->MCR = 0x04;     /* stop timer on match */
    LPC_TIMER2->TCR = 0x01;     /* start timer */
  
    /* wait until delay time has elapsed */
    while (LPC_TIMER2->TCR & 0x01);
  }
  else if ( timer_num == 3 )
  {
    LPC_TIMER3->TCR = 0x02;     /* reset timer */
    LPC_TIMER3->PR  = 0x00;     /* set prescaler to zero */
#ifdef LPC43XX
    LPC_TIMER3->MR[0] = delayInMs * ((M4Frequency / 1000) - 1);
#endif
#ifdef LPC18XX
    LPC_TIMER3->MR[0] = delayInMs * ((M3Frequency / 1000) - 1);
#endif
    LPC_TIMER3->IR  = 0xff;     /* reset all interrrupts */
    LPC_TIMER3->MCR = 0x04;     /* stop timer on match */
    LPC_TIMER3->TCR = 0x01;     /* start timer */
  
    /* wait until delay time has elapsed */
    while (LPC_TIMER3->TCR & 0x01);
  }
  return;
}

/******************************************************************************
** Function name:       Timer0_IRQHandler
**
** Descriptions:        Timer/Counter 0 interrupt handler
**
** parameters:          None
** Returned value:      None
** 
******************************************************************************/
void TIMER0_IRQHandler (void) 
{
  if ( LPC_TIMER0->IR & (0x1<<0) )
  {
    LPC_TIMER0->IR = 0x1<<0;        /* clear interrupt flag */
    timer0_m0_counter++;
  }
  if ( LPC_TIMER0->IR & (0x1<<1) )
  {
    LPC_TIMER0->IR = 0x1<<1;        /* clear interrupt flag */
    timer0_m1_counter++;
  }
  if ( LPC_TIMER0->IR & (0x1<<4) )
  {
    LPC_TIMER0->IR = 0x1<<4;        /* clear interrupt flag */
    timer0_capture0++;
  }
  if ( LPC_TIMER0->IR & (0x1<<5) )
  {
    LPC_TIMER0->IR = 0x1<<5;        /* clear interrupt flag */
    timer0_capture1++;
  }
  return;
}

/******************************************************************************
** Function name:       Timer1_IRQHandler
**
** Descriptions:        Timer/Counter 1 interrupt handler
**
** parameters:          None
** Returned value:      None
** 
******************************************************************************/
void TIMER1_IRQHandler (void)  
{  
  if ( LPC_TIMER1->IR & (0x1<<0) )
  {
    LPC_TIMER1->IR = 0x1<<0;        /* clear interrupt flag */
    timer1_m0_counter++;
  }
  if ( LPC_TIMER1->IR & (0x1<<1) )
  {
    LPC_TIMER1->IR = 0x1<<1;        /* clear interrupt flag */
    timer1_m1_counter++;
  }
  if ( LPC_TIMER1->IR & (0x1<<4) )
  {
    LPC_TIMER1->IR = 0x1<<4;        /* clear interrupt flag */
    timer1_capture0++;
  }
  if ( LPC_TIMER1->IR & (0x1<<5) )
  {
    LPC_TIMER1->IR = 0x1<<5;        /* clear interrupt flag */
    timer1_capture1++;
  }
  return;
}

/******************************************************************************
** Function name:       Timer2_IRQHandler
**
** Descriptions:        Timer/Counter 2 interrupt handler
**
** parameters:          None
** Returned value:      None
** 
******************************************************************************/
void TIMER2_IRQHandler (void)  
{  
  if ( LPC_TIMER2->IR & (0x1<<0) )
  {
    LPC_TIMER2->IR = 0x1<<0;        /* clear interrupt flag */
    timer2_m0_counter++;
  }
  if ( LPC_TIMER2->IR & (0x1<<1) )
  {
    LPC_TIMER2->IR = 0x1<<1;        /* clear interrupt flag */
    timer2_m1_counter++;
  }
  if ( LPC_TIMER2->IR & (0x1<<4) )
  {
    LPC_TIMER2->IR = 0x1<<4;        /* clear interrupt flag */
    timer2_capture0++;
  }
  if ( LPC_TIMER2->IR & (0x1<<5) )
  {
    LPC_TIMER2->IR = 0x1<<5;        /* clear interrupt flag */
    timer2_capture1++;
  }
  return;
}

/******************************************************************************
** Function name:       Timer3_IRQHandler
**
** Descriptions:        Timer/Counter 3 interrupt handler
**
** parameters:          None
** Returned value:      None
** 
******************************************************************************/
void TIMER3_IRQHandler (void)  
{  
  if ( LPC_TIMER3->IR & (0x1<<0) )
  {
    LPC_TIMER3->IR = 0x1<<0;        /* clear interrupt flag */
    timer3_m0_counter++;
  }
  if ( LPC_TIMER3->IR & (0x1<<1) )
  {
    LPC_TIMER3->IR = 0x1<<1;        /* clear interrupt flag */
    timer3_m1_counter++;
  }
  if ( LPC_TIMER3->IR & (0x1<<4) )
  {
    LPC_TIMER3->IR = 0x1<<4;        /* clear interrupt flag */
    timer3_capture0++;
  }
  if ( LPC_TIMER3->IR & (0x1<<5) )
  {
    LPC_TIMER3->IR = 0x1<<5;        /* clear interrupt flag */
    timer3_capture1++;
  }
  return;
}

/******************************************************************************
** Function name:       enable_timer
**
** Descriptions:        Enable timer
**
** parameters:          timer number: 0 or 1 or 2 or 3
** Returned value:      None
** 
******************************************************************************/
void enable_timer( uint8_t timer_num )
{
  if ( timer_num == 0 )
  {
    LPC_TIMER0->TCR = 1;
  }
  else if ( timer_num == 1 )
  {
    LPC_TIMER1->TCR = 1;
  }
  else if ( timer_num == 2 )
  {
    LPC_TIMER2->TCR = 1;
  }
  else if ( timer_num == 3 )
  {
    LPC_TIMER3->TCR = 1;
  }
  return;
}

/******************************************************************************
** Function name:       disable_timer
**
** Descriptions:        Disable timer
**
** parameters:          timer number: 0 or 1 oe 2 or 3
** Returned value:      None
** 
******************************************************************************/
void disable_timer( uint8_t timer_num )
{
  if ( timer_num == 0 )
  {
    LPC_TIMER0->TCR = 0;
  }
  else if ( timer_num == 1 )
  {
    LPC_TIMER1->TCR = 0;
  }
  else if ( timer_num == 2 )
  {
    LPC_TIMER2->TCR = 0;
  }
  else if ( timer_num == 3 )
  {
    LPC_TIMER2->TCR = 0;
  }
  return;
}

/******************************************************************************
** Function name:       reset_timer
**
** Descriptions:        Reset timer
**
** parameters:          timer number: 0 or 1 or 2 or 3
** Returned value:      None
** 
******************************************************************************/
void reset_timer( uint8_t timer_num )
{
  uint32_t regVal;

  if ( timer_num == 0 )
  {
    regVal = LPC_TIMER0->TCR;
    regVal |= 0x02;
    LPC_TIMER0->TCR = regVal;
  }
  else if ( timer_num == 1 )
  {
    regVal = LPC_TIMER1->TCR;
    regVal |= 0x02;
    LPC_TIMER1->TCR = regVal;
  }
  else if ( timer_num == 2 )
  {
    regVal = LPC_TIMER2->TCR;
    regVal |= 0x02;
    LPC_TIMER2->TCR = regVal;
  }
  else if ( timer_num == 3 )
  {
    regVal = LPC_TIMER3->TCR;
    regVal |= 0x02;
    LPC_TIMER3->TCR = regVal;
  }
  return;
}

#if 0
/******************************************************************************
** Function name:       Timer0_SetupIO
**
** Descriptions:        Setup IOCON and LOC register for CAPx and MATx
**
** parameters:          location
** Returned value:      None
** 
******************************************************************************/
void Timer0_SetupIO( uint32_t location )
{
  if ( location == 0 )
  {
    LPC_IOCON->P3_23           &= ~0x07;
    LPC_IOCON->P3_23           |= 0x03;     /* Timer0_16 CAP0 */
    LPC_IOCON->LOC_T0_CAP_0    = 0;
    LPC_IOCON->P3_24           &= ~0x07;
    LPC_IOCON->P3_24           |= 0x03;     /* Timer0_16 CAP1 */
    LPC_IOCON->LOC_T0_CAP_1    = 0;
    LPC_IOCON->P3_25           &= ~0x07;    
    LPC_IOCON->P3_25           |= 0x03;     /* Timer0_16 MAT0 */
    LPC_IOCON->P3_26           &= ~0x07;
    LPC_IOCON->P3_26           |= 0x03;     /* Timer0_16 MAT1 */
  }
  else if ( location == 1 ) /* This is location 1 for timer 0 */
  {
    LPC_IOCON->P1_26           &= ~0x07;    /*  Timer0_16 I/O config */
    LPC_IOCON->P1_26           |= 0x03;     /* Timer0_16 CAP0 */
    LPC_IOCON->LOC_T0_CAP_0    = 1;
    LPC_IOCON->P1_27           &= ~0x07;    /*  Timer0_16 I/O config */
    LPC_IOCON->P1_27           |= 0x03;     /* Timer0_16 CAP1 */
    LPC_IOCON->LOC_T0_CAP_1    = 1;
    LPC_IOCON->P1_28           &= ~0x07;    
    LPC_IOCON->P1_28           |= 0x03;     /* Timer0_16 MAT0 */
    LPC_IOCON->P1_29           &= ~0x07;
    LPC_IOCON->P1_29           |= 0x03;     /* Timer0_16 MAT1 */
  }
  return;
}

/******************************************************************************
** Function name:       Timer1_SetupIO
**
** Descriptions:        Setup IOCON and LOC register for CAPx and MATx
**
** parameters:          location
** Returned value:      None
** 
******************************************************************************/
void Timer1_SetupIO( uint32_t location )
{
  if ( location == 0 )
  {
    LPC_IOCON->P1_18           &= ~0x07;
    LPC_IOCON->P1_18           |= 0x03;     /* Timer1_16 CAP0 */
    LPC_IOCON->LOC_T1_CAP_0    = 0;
    LPC_IOCON->P3_28           &= ~0x07;
    LPC_IOCON->P3_28           |= 0x03;     /* Timer1_16 CAP1 */
    LPC_IOCON->LOC_T1_CAP_1    = 0;
    LPC_IOCON->P1_22           &= ~0x07;    
    LPC_IOCON->P1_22           |= 0x03;     /* Timer1_16 MAT0 */
    LPC_IOCON->P1_25           &= ~0x07;
    LPC_IOCON->P1_25           |= 0x03;     /* Timer1_16 MAT1 */
  }
  else if ( location == 1 ) /* This is location 1 for timer 1 */
  {
    LPC_IOCON->P3_27           &= ~0x07;        /*  Timer1_16 I/O config */
    LPC_IOCON->P3_27           |= 0x03;     /* Timer1_16 CAP0 */
    LPC_IOCON->LOC_T1_CAP_0    = 1;
    LPC_IOCON->P1_19           &= ~0x07;
    LPC_IOCON->P1_19           |= 0x03;     /* Timer1_16 CAP1 */
    LPC_IOCON->LOC_T1_CAP_1    = 1;
    LPC_IOCON->P3_29           &= ~0x07;    
    LPC_IOCON->P3_29           |= 0x03;     /* Timer1_16 MAT0 */
    LPC_IOCON->P3_30           &= ~0x07;
    LPC_IOCON->P3_30           |= 0x03;     /* Timer1_16 MAT1 */
  }
  return;
}

/******************************************************************************
** Function name:       Timer2_SetupIO
**
** Descriptions:        Setup IOCON and LOC register for CAPx and MATx
**
** parameters:          location
** Returned value:      None
** 
******************************************************************************/
void Timer2_SetupIO( uint32_t location )
{
  if ( location == 0 )
  {
    LPC_IOCON->P2_14           &= ~0x07;
    LPC_IOCON->P2_14           |= 0x03;     /* Timer2_32 CAP0 */
    LPC_IOCON->LOC_T2_CAP_0    = 0;
    LPC_IOCON->P2_15           &= ~0x07;
    LPC_IOCON->P2_15           |= 0x03;     /* Timer2_32 CAP1 */
    LPC_IOCON->LOC_T2_CAP_1    = 0;
    LPC_IOCON->P2_5            &= ~0x07;    
    LPC_IOCON->P2_5            |= 0x03;     /* Timer2_32 MAT0 */
    LPC_IOCON->P2_4            &= ~0x07;
    LPC_IOCON->P2_4            |= 0x03;     /* Timer2_32 MAT1 */
    LPC_IOCON->P2_3            &= ~0x07;
    LPC_IOCON->P2_3            |= 0x03;     /* Timer2_32 MAT2 */
    LPC_IOCON->P2_2            &= ~0x07;
    LPC_IOCON->P2_2            |= 0x03;     /* Timer2_32 MAT3 */
  }
  else if ( location == 1 ) /* This is location 1 for timer 2 */
  {
    LPC_IOCON->P2_6            &= ~0x07;        /*  Timer2_32 I/O config */
    LPC_IOCON->P2_6            |= 0x03;     /* Timer2_32 CAP0 */
    LPC_IOCON->LOC_T2_CAP_0    = 1;
    LPC_IOCON->P0_5            &= ~0x07;
    LPC_IOCON->P0_5            |= 0x03;     /* Timer2_32 CAP1 */
    LPC_IOCON->LOC_T2_CAP_1    = 1;
    LPC_IOCON->P0_6            &= ~0x07;    
    LPC_IOCON->P0_6            |= 0x03;     /* Timer2_32 MAT0 */
    LPC_IOCON->P0_7            &= ~0x07;
    LPC_IOCON->P0_7            |= 0x03;     /* Timer2_32 MAT1 */
    LPC_IOCON->P0_8            &= ~0x07;
    LPC_IOCON->P0_8            |= 0x03;     /* Timer2_32 MAT2 */
    LPC_IOCON->P0_9            &= ~0x07;
    LPC_IOCON->P0_9            |= 0x03;     /* Timer2_32 MAT3 */
  }
  else if ( location == 2 ) /* This is location 2 for timer 2 */
  {
    LPC_IOCON->P0_4            &= ~0x07;        /*  Timer2_32 I/O config */
    LPC_IOCON->P0_4            |= 0x03;     /* Timer2_32 CAP0 */
    LPC_IOCON->LOC_T2_CAP_0    = 2;
    LPC_IOCON->P4_28           &= ~0x07;    
    LPC_IOCON->P4_28           |= 0x03;     /* Timer2_32 MAT0 */
    LPC_IOCON->P4_29           &= ~0x07;
    LPC_IOCON->P4_29           |= 0x03;     /* Timer2_32 MAT1 */
    LPC_IOCON->P5_0            &= ~0x07;
    LPC_IOCON->P5_0            |= 0x03;     /* Timer2_32 MAT2 */
    LPC_IOCON->P5_1            &= ~0x07;
    LPC_IOCON->P5_1            |= 0x03;     /* Timer2_32 MAT3 */
  }
  else if ( location == 3 ) /* This is location 3 for timer 2 */
  {
    LPC_IOCON->P1_14           &= ~0x07;        /*  Timer2_32 I/O config */
    LPC_IOCON->P1_14           |= 0x03;     /* Timer2_32 CAP0 */
    LPC_IOCON->LOC_T2_CAP_0    = 3;
  }
  return;
}

/******************************************************************************
** Function name:       Timer3_SetupIO
**
** Descriptions:        Setup IOCON and LOC register for CAPx and MATx
**
** parameters:          location
** Returned value:      None
** 
******************************************************************************/
void Timer3_SetupIO( uint32_t location )
{
  if ( location == 0 )
  {
    LPC_IOCON->P0_23           &= ~0x07;
    LPC_IOCON->P0_23           |= 0x03;     /* Timer3_32 CAP0 */
    LPC_IOCON->LOC_T3_CAP_0    = 0;
    LPC_IOCON->P0_24           &= ~0x07;
    LPC_IOCON->P0_24           |= 0x03;     /* Timer3_32 CAP1 */
    LPC_IOCON->LOC_T3_CAP_1    = 0;
    LPC_IOCON->P0_10           &= ~0x07;    
    LPC_IOCON->P0_10           |= 0x03;     /* Timer3_32 MAT0 */
    LPC_IOCON->P0_11           &= ~0x07;
    LPC_IOCON->P0_11           |= 0x03;     /* Timer3_32 MAT1 */
    LPC_IOCON->P5_2            &= ~0x07;
    LPC_IOCON->P5_2            |= 0x03;     /* Timer3_32 MAT2 */
    LPC_IOCON->P5_4            &= ~0x07;
    LPC_IOCON->P5_4            |= 0x03;     /* Timer3_32 MAT3 */
  }
  else if ( location == 1 ) /* This is location 1 for timer 3 */
  {
    LPC_IOCON->P2_22           &= ~0x07;        /*  Timer3_32 I/O config */
    LPC_IOCON->P2_22           |= 0x03;     /* Timer3_32 CAP0 */
    LPC_IOCON->LOC_T3_CAP_0    = 1;
    LPC_IOCON->P2_23           &= ~0x07;
    LPC_IOCON->P2_23           |= 0x03;     /* Timer3_32 CAP1 */
    LPC_IOCON->LOC_T3_CAP_1    = 1;
    LPC_IOCON->P2_26           &= ~0x07;    
    LPC_IOCON->P2_26           |= 0x03;     /* Timer3_32 MAT0 */
    LPC_IOCON->P2_27           &= ~0x07;
    LPC_IOCON->P2_27           |= 0x03;     /* Timer3_32 MAT1 */
    LPC_IOCON->P2_30           &= ~0x07;
    LPC_IOCON->P2_30           |= 0x03;     /* Timer3_32 MAT2 */
    LPC_IOCON->P2_31            &= ~0x07;
    LPC_IOCON->P2_31            |= 0x03;        /* Timer3_32 MAT3 */
  }
  else if ( location == 2 ) /* This is location 2 for timer 3 */
  {
    LPC_IOCON->P1_10            &= ~0x07;       /*  Timer3_32 I/O config */
    LPC_IOCON->P1_10            |= 0x03;        /* Timer3_32 CAP0 */
    LPC_IOCON->LOC_T3_CAP_0     = 2;
    LPC_IOCON->P1_0             &= ~0x07;       
    LPC_IOCON->P1_0             |= 0x03;        /* Timer3_32 CAP1 */
    LPC_IOCON->LOC_T3_CAP_1     = 2;
    LPC_IOCON->P1_9             &= ~0x07;   
    LPC_IOCON->P1_9             |= 0x03;        /* Timer3_32 MAT0 */
    LPC_IOCON->P1_8             &= ~0x07;
    LPC_IOCON->P1_8             |= 0x03;        /* Timer3_32 MAT1 */
    LPC_IOCON->P1_4             &= ~0x07;
    LPC_IOCON->P1_4             |= 0x03;        /* Timer3_32 MAT2 */
    LPC_IOCON->P1_1             &= ~0x07;
    LPC_IOCON->P1_1             |= 0x03;        /* Timer3_32 MAT3 */
  }
  return;
}

#endif

/******************************************************************************
** Function name:       init_timer
**
** Descriptions:        Initialize timer, set timer interval, reset timer,
**                      install timer interrupt handler
**
** parameters:          timer number and timer interval
** Returned value:      true or false, if the interrupt handler can't be
**                      installed, return false.
** 
******************************************************************************/
uint32_t init_timer ( uint8_t timer_num, uint32_t location, uint32_t TimerInterval ) 
{
  if ( timer_num == 0 )
  {
    timer0_m0_counter = 0;
    timer0_m1_counter = 0;
    timer0_capture0 = 0;
    timer0_capture1 = 0;
//  LPC_SC->PCONP |= (0x01<<1);
//  Timer0_SetupIO( location );
    LPC_TIMER0->IR = 0x0F;          /* Clear MATx interrupt include DMA request */ 
    LPC_TIMER0->PR  = MHZ_PRESCALE; /* set prescaler to get 1 M counts/sec */
    LPC_TIMER0->MR[0] = TIME_INTERVALmS * 10; /* Set up 10 mS interval */
    LPC_TIMER0->MR[1] = TIME_INTERVALmS * 10; /* Set up 10 mS interval */
#if TIMER_MATCH
    LPC_TIMER0->EMR &= ~(0xFF<<4);
    LPC_TIMER0->EMR |= ((0x3<<4)|(0x3<<6));
#else
    /* Capture 0 and 1 on rising edge, interrupt enable. */
    LPC_TIMER0->CCR = (0x1<<0)|(0x1<<2)|(0x1<<3)|(0x1<<5);
#endif
    LPC_TIMER0->MCR = (0x3<<0)|(0x3<<3);    /* Interrupt and Reset on MR0 and MR1 */
    NVIC_EnableIRQ(TIMER0_IRQn);
    return (TRUE);
  }
  else if ( timer_num == 1 )
  {
    timer1_m0_counter = 0;
    timer1_m1_counter = 0;
    timer1_capture0 = 0;
    timer1_capture1 = 0;
//  LPC_SC->PCONP |= (0x1<<2);
//  Timer1_SetupIO( location );
    LPC_TIMER1->IR = 0x0F;          /* Clear MATx interrupt include DMA request */ 
    LPC_TIMER1->PR  = MHZ_PRESCALE; /* set prescaler to get 1 M counts/sec */
    LPC_TIMER1->MR[0] = TIME_INTERVALmS * 10; /* Set up 10 mS interval */
    LPC_TIMER1->MR[1] = TIME_INTERVALmS * 10; /* Set up 10 mS interval */
#if TIMER_MATCH
    LPC_TIMER1->EMR &= ~(0xFF<<4);
    LPC_TIMER1->EMR |= ((0x3<<4)|(0x3<<6));
#else
    /* Capture 0/1 on rising edge, interrupt enable. */
    LPC_TIMER1->CCR = (0x1<<0)|(0x1<<2)|(0x1<<3)|(0x1<<5);
#endif
    LPC_TIMER1->MCR = (0x3<<0)|(0x3<<3);    /* Interrupt and Reset on MR0 and MR1 */
    NVIC_EnableIRQ(TIMER1_IRQn);
    return (TRUE);
  }
  else if ( timer_num == 2 )
  {
    timer2_m0_counter = 0;
    timer2_m1_counter = 0;
    timer2_capture0 = 0;
    timer2_capture1 = 0;
//  LPC_SC->PCONP |= (0x01<<22);
//  Timer2_SetupIO( location );
#ifdef LPC43XX
    scu_pinmux(0x1,3,MD_PUP,FUNC1);
    scu_pinmux(0x1,4,MD_PUP,FUNC1);
    scu_pinmux(0x1,5,MD_PUP,FUNC1);
    scu_pinmux(0x1,9,MD_PUP,FUNC2);
#endif

#ifdef LPC18XX
    scu_pinmux(0x1,3,MD_PUP,FUNC1);
    scu_pinmux(0x1,4,MD_PUP,FUNC1);
    scu_pinmux(0x1,5,MD_PUP,FUNC1);
    scu_pinmux(0x1,9,MD_PUP,FUNC2);
#endif
    LPC_TIMER2->IR = 0x0F;          /* Clear MATx interrupt include DMA request */ 
//  LPC_TIMER2->PR  = MHZ_PRESCALE; /* set prescaler to get 1 M counts/sec */
    LPC_TIMER2->MR[0] = TimerInterval; /* Set up 10 mS interval */
    LPC_TIMER2->MR[1] = TimerInterval; /* Set up 10 mS interval */
#if TIMER_MATCH
    LPC_TIMER2->EMR &= ~(0xFF<<4);
    LPC_TIMER2->EMR |= ((0x3<<4)|(0x3<<6)|(0x3<<8)|(0x3<<10));
#else
    /* Capture 0/1 on rising edge, interrupt enable. */
    LPC_TIMER2->CCR = (0x1<<0)|(0x1<<2)|(0x1<<3)|(0x1<<5);
#endif
//  LPC_TIMER2->MCR = (0x3<<0)|(0x3<<3);    /* Interrupt and Reset on MR0 and MR1 */
    LPC_TIMER2->MCR = (0x2<<0)|(0x2<<3);    /* Reset on MR0 and MR1 */
//  NVIC_EnableIRQ(TIMER2_IRQn);
    return (TRUE);
  }
  else if ( timer_num == 3 )
  {
    timer3_m0_counter = 0;
    timer3_m1_counter = 0;
    timer3_capture0 = 0;
    timer3_capture1 = 0;
//  LPC_SC->PCONP |= (0x01<<23);
//  Timer3_SetupIO( location );
    LPC_TIMER3->IR = 0x0F;          /* Clear MATx interrupt include DMA request */ 
    LPC_TIMER3->PR  = MHZ_PRESCALE; /* set prescaler to get 1 M counts/sec */
    LPC_TIMER3->MR[0] = TimerInterval; /* Set up 10 mS interval */
    LPC_TIMER3->MR[1] = TimerInterval; /* Set up 10 mS interval */
#if TIMER_MATCH
    LPC_TIMER3->EMR &= ~(0xFF<<4);
    LPC_TIMER3->EMR |= ((0x3<<4)|(0x3<<6)|(0x3<<8)|(0x3<<10));
#else
    /* Capture 0/1 on rising edge, interrupt enable. */
    LPC_TIMER3->CCR = (0x1<<0)|(0x1<<2)|(0x1<<3)|(0x1<<5);
#endif
    LPC_TIMER3->MCR = (0x3<<0)|(0x3<<3);    /* Interrupt and Reset on MR0 and MR1 */
    NVIC_EnableIRQ(TIMER3_IRQn);
    return (TRUE);
  }
  return (FALSE);
}


